<?php

/*
 * Model class for UserMilestone
 * This class handles user milestones, like when it is completed, invalidate scenarios etc.,
 *
 * @author Shiva <shiva.dharana@univariety.com>
 * @version 1.0
 * @package sms.models
 */

Yii::import('application.modules.sms.models.base.BaseUserMilestone');

class UserMilestone extends BaseUserMilestone{

    /**
     * User miletone statuses
     * @var constant
     */
    const STATUS_INPROGRESS = 'InProgress';
    const STATUS_CONFIRMED = 'Confirmed';
    const STATUS_INVALIDATED = 'Invalidated';

    static $milestones = array(
            '1' => 'Hypothesis',
            '2' => 'Finalisation',
            '3' => 'Test Preparation',
            '4' => 'Ideal List',
            '5' => 'Shortlist',
            '6' => 'Outplacement',
    );

    public static function model($className=__CLASS__){
        return parent::model($className);
    }

    /**
     * @return array validation rules for model attributes.
     */
    public function rules(){
        // NOTE: you should only define rules for those attributes that
        // will receive user inputs.
        return array(
        array('user_ref_id, milestone_id', 'required'),
        array('milestone_id', 'numerical', 'integerOnly'=>true),
        array('user_ref_id', 'length', 'max'=>10),
        // The following rule is used by search().
        // Please remove those attributes that should not be searched.
        array('user_milestone_id, user_ref_id, milestone_id, status, added_on', 'safe', 'on'=>'search'),
        );
    }

    /**
     * @return array relational rules.
     */
    public function relations(){
        // NOTE: you may need to adjust the relation name and the related
        // class name for the relations automatically generated below.
        return array(
			'user' => array(self::BELONGS_TO, 'TblUserMaster', 'user_ref_id'),
        );
    }

    protected function beforeSave(){
        if(parent::beforeSave()){
            if(empty($this->added_on))
                $this->added_on = date('Y-m-d H:i:s');
            return true;
        }
        return false;
    }

    /**
     * checkNSave will check for existing record, if already exists, updates it else creates a new record
     * @param bool $validate
     * @return bool
     */
    public function checkNSave($validate = false){

        $result = self::model()->findByAttributes(array(
            'user_ref_id' => $this->user_ref_id,
            'milestone_id' => $this->milestone_id,
            'status' => array(self::STATUS_CONFIRMED, self::STATUS_INPROGRESS),
        ));

        // update existing record.
        if($result){
            $result->status = self::STATUS_CONFIRMED;
            $result->added_on = date('Y-m-d H:i:s');
            return $result->update(array(
            	'status',
                'added_on')
            );
        }else{
            return $this->save($validate);
        }
    }

    /**
     * Gets the user milestone confirmation status.
     * @param int $userId
     * @param int $milestoneId
     * @return bool
     */
    public function getConfirmationStatus($userId, $milestoneId){
        $row = self::model()->findByAttributes(array(
            'user_ref_id' => $userId,
            'milestone_id' => $milestoneId,
            'status' => self::STATUS_CONFIRMED,
        ));
        if($row){
            return true;
        }
        return false;
    }

    /**
     * Returns true if the milestone has been confirmed, else false is returned
     * @param int $userId
     * @param int $milestoneId
     * @return bool
     */
    public static function isConfirmed($userId, $milestoneId){
        $row = self::model()->findByAttributes(array(
            'user_ref_id' => $userId,
            'milestone_id' => $milestoneId,
            'status' => self::STATUS_CONFIRMED,
        ));
        if($row){
            return true;
        }
        return false;
    }

    /**
     * Marks a user milestone as confirmed
     * @param int $userId
     * @param int $milestoneId
     * @return bool
     */
    public static function confirmMilestone($userId, $milestoneId, $date = null){
        $userMilestone = self::model()->findByAttributes(array(
            'user_ref_id' => $userId,
            'milestone_id' => $milestoneId,
            'status' => array(self::STATUS_CONFIRMED, self::STATUS_INPROGRESS),
        ));
        if($userMilestone){
            $userMilestone->status = self::STATUS_CONFIRMED;
            return $userMilestone->save();
        }
        //
        $userMilestone = new UserMilestone;
        $userMilestone->user_ref_id = $userId;
        $userMilestone->milestone_id = $milestoneId;
        $userMilestone->status = self::STATUS_CONFIRMED;
        if(!empty($date))
            $userMilestone->added_on = $date;
        return $userMilestone->save();
    }

    /**
     * Resets a user milestone.
     * All the confirmed, started milestones greater this milestone will also be reset.
     * @return bool
     */
    public function invalidateMilestone(){
        $bl = true;
        for($i = $this->milestone_id; $i <= 6; ++$i){
            $userMilestone = self::model()->findByAttributes(array(
                'user_ref_id' => $this->user_ref_id,
                'milestone_id' => $i,
                'status' => array(self::STATUS_CONFIRMED, self::STATUS_INPROGRESS),
            ));
            if($userMilestone){
                $userMilestone->status = self::STATUS_INVALIDATED;
                $bl = $userMilestone->delete();
            }
            if($i == 4){
                $this->invalidateMileStone4();
            }

            if($i == 3){
                $this->invalidateMilestone3();
            }
            
            if($i == 5){
                $this->invalidateMilestone5();
            }

            //Invalidate ms6 sub steps
            if($i == 6){
                for($j = 61; $j < 64; ++$j){
                    $userMilestone = self::model()->findByAttributes(array(
                        'user_ref_id' => $this->user_ref_id,
                        'milestone_id' => $j,
                        'status' => array(self::STATUS_CONFIRMED, self::STATUS_INPROGRESS),
                    ));
                    if($userMilestone){
                        $userMilestone->status = self::STATUS_INVALIDATED;
                        $bl = $userMilestone->save();
                    }
                }

            }
        }
        return $bl;
    }

    /**
     * Sets student ideal list as invalid
     * If student country has changed, delete all the records.
     * If student department has changed, deletes only the removed department lists
     * @param bool $resetDepartmentsOnly, Optional, default false, if set to true, lists with removed departments will be deleted, rest will be untouched. If set to false all the lists will be deleted
     * @return bool
     */
    public function invalidateMileStone4($resetDepartmentsOnly = false){

        //Find the courses which have been changed and invalidate them
        // fetch course preferences
        $preferenceDepartments = array();
        if($resetDepartmentsOnly === true){
            $departments = $this->user->studentPreferredDepartments();
            $preferenceDepartments = array();
            foreach($departments as $department){
                $preferenceDepartments[] = $department->sub_dept_ref_id;
            }
        }

        $lists = StudentIdealList::model()->findAll(array(
        	'condition' => 'status != :status AND user_ref_id = :user_ref_id',
            'params'=>array(':status' => StudentIdealList::STATUS_INVALID, ':user_ref_id' => $this->user_ref_id),
        ));
        foreach($lists as $list){
            if(!($resetDepartmentsOnly && in_array($list->department_ref_id, $preferenceDepartments))){
                $list->status = StudentIdealList::STATUS_INVALID;
                $list->save();
            }
        }
        return true;
    }

    /**
     * Finds whether a student has attended a workshop or not.
     * @param int $studentId, user_id of the student
     * @param int $workshopId, index number of workshop, from 1 to 6, has nothing to do with primary key of workshop_master
     * @return bool, Returns true if attended else false
     */
    public static function isStudentWorkshopCompleted($studentId, $workshopId){
        $sql = <<<EOT
        select 
        	count(*)
        from 
        	workshop_si_group_student as a 
        	join workshop_si_group as d on a.workshop_si_group_ref_id = a.workshop_si_group_ref_id
        	join workshop_si as c on d.workshop_si_ref_id = c.workshop_si_id
        	join workshop_master as b on b.workshop_master_id = c.workshop_master_ref_id
        where 
        	a.user_ref_id = :studentId
        	AND b.workshop_name = :workshopName
EOT;
        $workshop = 'workshop' . $workshopId;


        $cnt = self::model()->countBySql($sql, array(':studentId' => $studentId, ':workshopName' => $workshop));
        if($cnt > 0){
            return true;
        }
        return false;
    }

    /**
     * Checks whether aspiration, bestfit lists in all departments are confirmed or not
     * @param int $studentId, student id
     * @return bool, If all the lists are confirmed returns true else returns false.
     */
    public static function areMs4ListsConfirmed($studentId){
        $departments = TblUserMaster::model()->findByPk($studentId)->studentPreferredDepartments();

        if(count($departments) == 0){
            return false;
        }

        $listsToCheck = array(StudentIdealList::TYPE_ASPIRATIONAL, StudentIdealList::TYPE_BEST_FIT);
        foreach($departments as $deparment){
            for($i = 0, $cnt = count($listsToCheck); $i < $cnt; ++$i){
                $list = StudentIdealList::getRecord($studentId, $deparment->sub_dept_ref_id, $listsToCheck[$i]);

                if($list->status != StudentIdealList::STATUS_CONFIRMED)
                return false;
            }
        }
        return true;
    }

    /**
     * Checks whether safe lists in all departments are confirmed or not.
     * @param TblUserMaster $student
     * @return bool, If all the lists are done returns true else returns false.
     */
    public static function areMs5ListsConfirmed($student){
        $departments = $student->studentPreferredDepartments();

        if(count($departments) == 0){
            return false;
        }

        $listsToCheck = array(StudentIdealList::TYPE_SAFE);
        foreach($departments as $deparment){
            $list = StudentIdealList::getRecord($student->user_id, $deparment->sub_dept_ref_id, StudentIdealList::TYPE_SAFE);
            if($list->status != StudentIdealList::STATUS_CONFIRMED)
                return false;
        }
        return true;
    }

    public static function getMilestones(){
        return self::$milestones;
    }

    public function getCareer($userId, $milestoneId){

    }
    
    /**
     * This method returns countries, prior to confirmation of milestone.
     * @param integer $studentId
     * @param integer $milestoneId
     */
    public static function getCountries($studentId, $milestoneId){
        //User has not completed the milstone.
        if(!self::isConfirmed($studentId, $milestoneId) && $milestoneId != 0){
            return array();
        }
        
        // if next milestone is not confirmed, return current preferences.
        if(!self::isConfirmed($studentId, $milestoneId + 1) || $milestoneId >= 2){
            $criteria = new CDbCriteria();
            $criteria->compare('user_ref_id', $studentId);
            $list = StudentPreferredCountry::model()->findAll($criteria);
            return $list;
        }
        
        $row = self::model()->findByAttributes(array(
            'user_ref_id' => $studentId,
            'milestone_id' => $milestoneId + 1,
            'status' => self::STATUS_CONFIRMED,
        ));
        $confirmedDate = $row->added_on;

        $criteria = new CDbCriteria();
        $criteria->compare('user_ref_id', $studentId);
        $criteria->compare('logged_for', 'MS' . $milestoneId . '-' . $confirmedDate);
        $list = StudentPreferredCountryLog::model()->findAll($criteria);
        return $list;
    }

    public static function getDepartments($studentId, $milestoneId){
        //User has not completed the milstone.
        if(!self::isConfirmed($studentId, $milestoneId) && $milestoneId != 0){
            return array();
        }
        // if next milestone is not confirmed, return current departments.
        if(!self::isConfirmed($studentId, $milestoneId + 1) || $milestoneId >= 2){
            $criteria = new CDbCriteria();
            $criteria->compare('user_ref_id', $studentId);
            $list = StudentPreferredDepartment::model()->findAll($criteria);
            return $list;
        }
        
        $row = self::model()->findByAttributes(array(
            'user_ref_id' => $studentId,
            'milestone_id' => $milestoneId + 1,
            'status' => self::STATUS_CONFIRMED,
        ));
        $confirmedDate = $row->added_on;

        $criteria = new CDbCriteria();
        $criteria->compare('user_ref_id', $studentId);
        $criteria->compare('logged_for', 'MS' . $milestoneId . '-' . $confirmedDate);
        $list = StudentPreferredDepartmentLog::model()->findAll($criteria);
        return $list;
    }

    public static function getCareers($studentId, $milestoneId){
        //User has not completed the milstone.
        if(!self::isConfirmed($studentId, $milestoneId)){
            return array();
        }

        // At the time of writing this code, there is no log for preferred career
        // so returning always the current list
        $student = TblUserMaster::model()->findByPk($studentId);
        return $student->studentPreferredCareers();
    }

    /**
     * Sets student tests and preparation as invalid
     * @return bool
     */
    public function invalidateMileStone3(){
        return StudentEntranceExamVendorBranch::model()->deleteAllByAttributes(array('user_ref_id' => $this->user_ref_id));
    }
    
    /**
     * Marks student ideal list safe colleges as not confirmed by user.
     * @return boolean
     */
    public function invalidateMilestone5(){
        $student = $this->user;
        $departments = $student->studentPreferredDepartments();

        if(count($departments) == 0){
            return false;
        }

        $listsToCheck = array(StudentIdealList::TYPE_SAFE);
        foreach($departments as $deparment){
            $list = StudentIdealList::getRecord($student->user_id, $deparment->sub_dept_ref_id, StudentIdealList::TYPE_SAFE);
            if($list->status == StudentIdealList::STATUS_CONFIRMED){
                $list->status = StudentIdealList::STATUS_UNIDONE;
                $list->save();
            }
        }
        return true;
    
    }
    
	/**
     * Checks whether ms3 vendors all departments are confirmed or not
     * @param integar $studentId, student id
     * @return boolean
     */
    public static function areMs3ListsConfirmed($studentId){
        $departments = TblUserMaster::model()->findByPk($studentId)->studentPreferredDepartments();

        if(count($departments) == 0){
            return false;
        }
        foreach($departments as $deparment){
            $bool = StudentEntranceExamVendorBranch::isStudentDepartmentVendorsConfirmed($studentId, $deparment->sub_dept_ref_id);
            if($bool === false){
                return false;
            }
        }
        return true;
    }

    public static function getMilestoneDetails($userId, $milestoneId) {
        $cdbCriteria = new CDbCriteria();
        $cdbCriteria->addCondition("user_ref_id = ".$userId." ", "AND");
        $cdbCriteria->addCondition("milestone_id = ".$milestoneId." ", "AND");
        $cdbCriteria->addCondition("status = 'Confirmed' ", "AND");
        $result = self::model()->find($cdbCriteria);
        return $result;

    }

    public static function getUserConfirmedMilestoneDetails($userId) {
        $cdbCriteria = new CDbCriteria();
        $cdbCriteria->addCondition("user_ref_id = ".$userId." ", "AND");
        $cdbCriteria->addCondition("status = 'Confirmed' ", "AND");
        $result = self::model()->findAll($cdbCriteria);
        return $result;
    }
    
    public function checkPreviousMilestoneStatus($userId, $currentMilestone) {
        if ($currentMilestone > 1) {
            for ($i = 1; $i < $currentMilestone;$i++) {
                if (UserMilestone::isConfirmed($userId, $i)) {

                } else {
                    return $i;
                }
            }
        }
    }
}